/* =============================================================================
	POJECT:		UKProgressPanel
	FILE:		UKProgressPanelTask.m
	PURPOSE:	A single "task" (i.e. progress bar and status text field) for
				our MT-Newswatcher/Finder-style progress window for keeping the
				user current on concurrently running tasks.
	AUTHORS:	M. Uli Kusterer (UK), (c) 2003, all rights reserved.
	
	REQUIRES:	UKProgressPanelTask.h
				UKProgressPanelTask.nib
				UKProgressPanel.h
				UKProgressPanel.m
				(UKProgressPanel.nib)
				(UKProgressPanel.strings)
	
	NOTE:		UKProgressPanel and UKProgressPanelTask are thread-safe.
	
	DIRECTIONS:
			Using the progress panel is very simple: Use newProgressPanelTask
			and consorts to create a new task when you begin your lengthy
			operation, and release it when you have finished.
			
			The task will automatically take care of creating and showing the
			progress panel if needed, and will add itself to the shared
			progress panel instance.
			
			A task supports most of the methods of NSProgressIndicator to
			allow you to control the progress bar without any deep-reaching
			code changes. Use setTitle: to provide a general title indicating
			what action this is (e.g. "Emptying the trash"). The title will be
			displayed in bold above the progress bar. Use setStatus: to provide
			information about the step currently being executed, expected time
			to finish etc. - This will be displayed below the progress bar.
   ========================================================================== */

/* -----------------------------------------------------------------------------
	Headers:
   -------------------------------------------------------------------------- */

#import "UKProgressPanelTask.h"
#import "UKProgressPanel.h"
#import "UKMainThreadProxy.h"
#include <unistd.h>


/* -----------------------------------------------------------------------------
	Macros:
   -------------------------------------------------------------------------- */

#ifndef UK_DEBUG_PRINT
#define	UK_DEBUG_PRINT(a,b)		//NSLog((a),(b))
#endif


extern NSThread*	gProgressPanelMainThread;


@implementation UKProgressPanelTask


/* -----------------------------------------------------------------------------
	newProgressPanelTask:
		Convenience constructor for creating a task. This automatically adds
		the task to the shared progress panel, creating one if necessary.
		
		Caller is responsible for releasing the result of this.
   -------------------------------------------------------------------------- */

+(id)	newProgressPanelTask
{
	UKProgressPanelTask*	el;
	
	el = [[self alloc] init];
	
	[[UKProgressPanel sharedProgressPanel] performSelectorOnMainThread: @selector(addProgressPanelTask:) withObject: el waitUntilDone: YES];
	
	return el;
}


/* -----------------------------------------------------------------------------
	Constructor:
		Do not call this unless you really know what you're doing.
   -------------------------------------------------------------------------- */

-(id)	init
{
	if( self = [super init] )
	{
		topLevelObjects = [[NSMutableArray alloc] init];
		NSDictionary*	ent = [NSDictionary dictionaryWithObjectsAndKeys:
									self, @"NSOwner",
									topLevelObjects, @"NSTopLevelObjects",
									nil];
		NSBundle*		mainB = [NSBundle mainBundle];
		[mainB loadNibFile: [self taskViewNibName]
							externalNameTable: ent withZone: [self zone]];	// We're responsible for releasing the top-level objects in the NIB (our view, right now).
		if( progressTaskView == nil )
		{
			NSLog(@"%@: Couldn't find NIB file \"%@\".", NSStringFromClass([self class]),[self taskViewNibName]);
			[self autorelease];
			return nil;
		}
		stopAction = @selector(stop:);
		stopped = NO;
		
		inUse = [[NSLock alloc] init];
	}
	
	return self;
}


-(void)	release
{
	if( [NSThread currentThread] == gProgressPanelMainThread )
		[super release];
	else
		[self performSelectorOnMainThread: @selector(release) withObject: nil waitUntilDone: NO];
}


/* -----------------------------------------------------------------------------
	Destructor:
		Makes sure we no longer belong to our progress window.
   -------------------------------------------------------------------------- */

-(void)	dealloc
{
	if( [NSThread currentThread] != gProgressPanelMainThread )
		NSLog(@"*** WARNING: UKProgressPanelTask got dealloced from another thread. Our release override should make that impossible?!");

	UK_DEBUG_PRINT(@"%lx: Waiting for removal of panel.", self);
	[inUse lock];
	[[UKProgressPanel sharedProgressPanel] removeProgressPanelTask: self];
	
	UK_DEBUG_PRINT(@"%lx: Panel removed.", self);
	
	[topLevelObjects makeObjectsPerformSelector: @selector(release)];
	[topLevelObjects release];
	topLevelObjects = nil;
	
	UK_DEBUG_PRINT(@"%lx: Toplevel objects removed.", self);
	
	[inUse release];
	inUse = nil;
	UK_DEBUG_PRINT(@"%lx: Lock removed.", self);
	
	UK_DEBUG_PRINT(@"%lx: got dealloced.", self);
	
	[super dealloc];
}


/* -----------------------------------------------------------------------------
	Controlling the progress bar:
		These methods simply forward the messages to our progress bar.
   -------------------------------------------------------------------------- */

-(double)		minValue								{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: retrieving minValue", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; double n = [pb minValue]; [pb release]; [inUse unlock]; return n; } else return 0; }
-(double)		maxValue								{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: retrieving maxValue", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; double n = [pb maxValue]; [pb release]; [inUse unlock]; return n; } else return 0; }
-(void)			setMinValue: (double)newMinimum			{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: Changing minValue", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; [pb setMinValue: newMinimum]; [pb display]; [pb release]; [inUse unlock]; } }
-(void)			setMaxValue: (double)newMaximum			{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: Changing maxValue", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; [pb setMaxValue: newMaximum]; [pb display]; [pb release]; [inUse unlock]; } }

-(double)		doubleValue								{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: retrieving value", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; double n = [pb doubleValue]; [pb release]; [inUse unlock]; return n; } else return 0; }
-(void)			setDoubleValue: (double)doubleValue		{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: Changing value", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; [pb setDoubleValue: doubleValue]; [pb display]; [pb release]; [inUse unlock]; } }
-(void)			incrementBy: (double)delta				{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: incrementing value", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; [pb incrementBy: delta]; [pb display]; [pb release]; [inUse unlock]; } }

-(BOOL)			isIndeterminate							{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: retrieving isIndeterminate", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; BOOL n = [pb isIndeterminate]; [pb release]; [inUse unlock]; return n; } else return 0; }				
-(void)			setIndeterminate: (BOOL)flag			{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: Changing isIndeterminate of", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; [pb setIndeterminate: flag]; [pb display]; [pb release]; [inUse unlock]; } }
-(void)			animate: (id)sender						{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: animating", self); NSProgressIndicator* pb = [progressBar copyMainThreadProxy]; [pb animate: sender]; [pb display]; [pb release]; [inUse unlock]; } }


/* -----------------------------------------------------------------------------
	title/status:
   -------------------------------------------------------------------------- */

-(void)			setTitle: (NSString*)title				{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: Changing title", self); NSTextField* tf = [progressTitleField copyMainThreadProxy]; [tf setStringValue: title]; [tf display]; [tf release]; [inUse unlock]; } }
-(void)			setStatus: (NSString*)status			{ if( [inUse tryLock] ) { UK_DEBUG_PRINT(@"%lx: Changing status", self); NSTextField* ts = [progressStatusField copyMainThreadProxy]; [ts setStringValue: status]; [ts display]; [ts release]; [inUse unlock]; } }



/* -----------------------------------------------------------------------------
	setStopDelegate:
		Use this to specify an object to be sent the stopAction (defaults to
		@selector(stop:)) when the user clicks the "Stop" button in this
		panel.
		
		This defaults to nil, meaning no notification will be sent.
   -------------------------------------------------------------------------- */

-(void)			setStopDelegate: (id)target
{
	stopDelegate = target;
}


/* -----------------------------------------------------------------------------
	stopDelegate:
		Returns the delegate that will be notified of clicks in the "Stop"
		button.
   -------------------------------------------------------------------------- */

-(id)			stopDelegate
{
	return stopDelegate;
}


/* -----------------------------------------------------------------------------
	setStopAction:
		Use this to specify the message (defaults to @selector(stop:)) to be
		sent to the stopDelegate when the user clicks the "Stop" button in this
		panel. If you don't specify a stop delegate, the stop button will be
		disabled. (I'm not hiding it since that's too much work in Cocoa and
		I want to encourage writing abortable operations).
   -------------------------------------------------------------------------- */

-(void)			setStopAction: (SEL)action
{
	stopAction = action;
}


/* -----------------------------------------------------------------------------
	stopAction:
		The message (defaults to @selector(stop:)) to be sent to the
		stopDelegate when the user clicks the "Stop" button in this panel.
   -------------------------------------------------------------------------- */

-(SEL)			stopAction
{
	return stopAction;
}


/* -----------------------------------------------------------------------------
	stopped:
		Accessor for the flag that is set by our "stop:" action method when the
		user clicks the "Stop" button.
   -------------------------------------------------------------------------- */

-(BOOL)			stopped
{
	return stopped;
}


/* -----------------------------------------------------------------------------
	stop:
		This method is called by the "Stop" button whenever it's been clicked.
		
		It does two things: It sets this object's "stopped" flag that you can
		check, and it sends the stopAction to the stopDelegate.
		
	REVISIONS:
		2004-10-17	UK	Changed this to send message to main thread and to
						pass "self" as the sender. The "stop" button isn't
						really interesting.
   -------------------------------------------------------------------------- */

-(IBAction)	stop: (id)sender
{
	[[progressStopButton mainThreadProxy] setEnabled: NO];
	stopped = YES;
	
	[stopDelegate performSelectorOnMainThread: stopAction withObject: self waitUntilDone: NO];
}


/* -----------------------------------------------------------------------------
	progressTaskView:
		Returns the view containing the title/status fields and progress bar
		for this task.
   -------------------------------------------------------------------------- */

-(NSView*)	progressTaskView
{
	return progressTaskView;
}


// Name of NIB file to use for our task:
-(NSString*)	taskViewNibName
{
	#if UK_PROGRESS_PANEL_SAFARI_STYLE
	return @"UKListProgressPanelTask";
	#else
	return @"UKProgressPanelTask";
	#endif
}

@end
