/* =============================================================================
	POJECT:		UKProgressPanel
	PURPOSE:	MT-Newswatcher/Finder-style progress window for keeping the
				user current on concurrently running tasks.
	AUTHORS:	M. Uli Kusterer (UK), (c) 2003, all rights reserved.
	
	REQUIRES:	UKProgressPanel.h
				UKProgressPanel.nib
				UKProgressPanel.strings
				UKProgressPanelTask.h
				UKProgressPanelTask.m
				(UKProgressPanelTask.nib)
   ========================================================================== */

/* -----------------------------------------------------------------------------
	Headers:
   -------------------------------------------------------------------------- */

#import "UKProgressPanel.h"
#import "UKProgressPanelTask.h"
#import "UKMainThreadProxy.h"


/* -----------------------------------------------------------------------------
	Globals:
   -------------------------------------------------------------------------- */

static UKProgressPanel*		gMainProgressPanel = nil;				// Here we keep track of our shared panel instance (singleton pattern).
NSLock*						gUKProgressPanelThreadLock = nil;		// Users will want to use threads with this. We need a mutex lock to avoid several progress panels and such stuff.


@implementation UKProgressPanel

+(void)	initialize
{
	gUKProgressPanelThreadLock = [[NSLock alloc] init];
}


/* -----------------------------------------------------------------------------
	sharedProgressPanel:
		Returns a pointer to our shared UKProgressPanel instance, creating
		it if none exists yet.
   -------------------------------------------------------------------------- */

+(UKProgressPanel*)	sharedProgressPanel
{
	[gUKProgressPanelThreadLock lock];
	if( !gMainProgressPanel )
		gMainProgressPanel = [[self alloc] init];
	[gUKProgressPanelThreadLock unlock];
	
	return gMainProgressPanel;
}


/* -----------------------------------------------------------------------------
	Constructor:
		Loads the progress window from UKProgressPanel.nib and slightly changes
		its behavior in ways that aren't possible through the NIB.
   -------------------------------------------------------------------------- */
		
/**		\note   This window's behavior has been chosen intentionally. It uses a
		utility window with small title bar since it's not associated with a
		document and isn't one itself, but it still performs the function of a
		palette.
		
		It is set not to hide when the application is in the background since
		the user may want to check whether the app has finished while it was
		in the background.
		
		It is also set to be at normal window level because the default level
		for utility windows is a system-wide floater, which would mean our
		progress window would obscure other apps' windows. It is also at normal
		window level to allow that the user send it behind another window when
		working.
*/

-(id)	init
{
	if( self = [super init] )
	{
		[NSBundle loadNibNamed: [self windowNibName] owner: self];	// We're responsible for releasing any top-level objects in the NIB (right now only our NSPanel).
		NSWindow*	tlw = [taskListWindow mainThreadProxy];
		[tlw setHidesOnDeactivate: NO];			// Allow checking on progress while app's in back.
		[tlw setLevel: NSNormalWindowLevel];	// Allow sending it behind documents.
		[tlw setReleasedWhenClosed: NO];		// Only hide on close box click.
		
		unretainedTasks = (NSMutableArray*) CFArrayCreateMutable( kCFAllocatorDefault, 0, NULL );	// This aray doesn't retain its objects.
		
		//[taskContentView setFlipped: YES];
	}
	
	return self;
}


/* -----------------------------------------------------------------------------
	Destructor:
		Note that you may *not* destruct this window while any tasks listed
		in it are still running. To avoid circular dependencies, this window
		does not know which tasks it contains. It does know about their content
		views, though.
   -------------------------------------------------------------------------- */

-(void)	dealloc
{
	[gUKProgressPanelThreadLock lock];
	[taskListWindow orderOut: nil];
	[taskListWindow release];
	
	gMainProgressPanel = nil;	// Make sure user can create a new shared instance if desired.
	[gUKProgressPanelThreadLock unlock];
	
	[unretainedTasks release];
	
	[super dealloc];
}


/* -----------------------------------------------------------------------------
	orderFront:
		Passes the message on to the task list panel.
   -------------------------------------------------------------------------- */

-(void)	orderFront: (id)sender
{
	[[taskListWindow mainThreadProxy] orderFront: sender];
}


/* -----------------------------------------------------------------------------
	stopAllTasks:
		Cancels all the tasks that are currently running.
   -------------------------------------------------------------------------- */

-(void)	stopAllTasks: (id)sender
{
	NSEnumerator*			enny = [unretainedTasks reverseObjectEnumerator];
	UKProgressPanelTask*	currObj = nil;
	
	while(( currObj = [enny nextObject] ))
		[currObj stop: self];
}


NSThread*	gProgressPanelMainThread = nil;


/* -----------------------------------------------------------------------------
	addProgressPanelTask:
		This is called by UKProgressPanelTasks when they are created. It adds
		the task's view to the list in the window above the current tasks.
		Then it brings the window to the front so the user sees that there's
		a new task in progress. Since the window can't become key, the user
		shouldn't be too annoyed by this, as keyboard focus etc. aren't
		changed.
		
		This also updates the "n tasks in progress..." message at the top
		of the window.
		
		Must be called on the main thread.
   -------------------------------------------------------------------------- */

-(void)	addProgressPanelTask: (UKProgressPanelTask*)theElement
{
	NSArray*			subs = [taskContentView subviews];
	NSView*				lastTaskView = [subs lastObject];
	NSSize				newSize;
	
	[taskContentView addSubview: [theElement progressTaskView]];
	
	if( lastTaskView )
	{
		NSRect	lastBox = [lastTaskView frame];
		
		// Position the new box above all others:
		lastBox.origin.y += lastBox.size.height;							// Move box up one slot.
		[[theElement progressTaskView] setFrameOrigin: lastBox.origin];		// Move new field to this position.
		
		// Calculate total height up to our new box:
		newSize = lastBox.size;
		newSize.height += lastBox.origin.y +1;
	}
	else
	{
		newSize = [[theElement progressTaskView] frame].size;
	}
	
	subs = [taskContentView subviews];	// Make sure we get a current list & current count.
	
	[taskContentView setFrameSize: newSize];				// Make content view that size.
	
	// Scroll new box into view:
	[taskContentView scrollRectToVisible: [[theElement progressTaskView] frame]];
	
	// Update "number of tasks" status display:
	[taskStatus setStringValue: [NSString stringWithFormat: NSLocalizedStringFromTable(@"%u tasks in progress...",@"UKProgressPanel", nil), [subs count]]];
	[taskStatus display];
	
	[taskListWindow display];
	[taskListWindow orderFront: self];
	
	[unretainedTasks addObject: theElement];
	
	if( !gProgressPanelMainThread )
		gProgressPanelMainThread = [NSThread currentThread];
}


/* -----------------------------------------------------------------------------
	removeProgressPanelTask:
		This is called by UKProgressPanelTasks when they are destroyed. It
		removes the task's view from the list in the window, moving down any
		views above it.
		
		This also updates the "n tasks in progress..." message at the top
		of the window.
		
		Must be called on the main thread.
   -------------------------------------------------------------------------- */

-(void)	removeProgressPanelTask: (UKProgressPanelTask*)theElement
{
	NSView*				elementsView = [theElement progressTaskView];
	NSArray*			subs = [taskContentView subviews];
	unsigned int		pos = [subs indexOfObject: elementsView];
	NSEnumerator*		elEnum = [subs objectEnumerator];
	NSSize				sizeGone = [elementsView frame].size;
	unsigned int		x;
	NSView*				currElemView = nil;
	
	[elementsView setNeedsDisplay:YES];
	[elementsView removeFromSuperview];
	
	// Update "number of tasks" status display:
	unsigned int tCount = [subs count];
	if( tCount == 0 )
		[taskStatus setStringValue: NSLocalizedStringFromTable(@"No active tasks.",@"UKProgressPanel", nil)];
	else
		[taskStatus setStringValue: [NSString stringWithFormat: NSLocalizedStringFromTable(@"%u tasks in progress...",@"UKProgressPanel", nil), tCount]];
	[taskStatus setNeedsDisplay:YES];
	
	// Move down elements above the one we're removing:
	for( x = 0; currElemView = [elEnum nextObject]; x++ )
	{
		if( x > pos )
		{
			NSPoint		currOrigin = [currElemView frame].origin;
			currOrigin.y -= sizeGone.height;
			[currElemView setFrameOrigin: currOrigin];
			[currElemView setNeedsDisplay:YES];
		}
	}
	
	[taskContentView setNeedsDisplay:YES];
	
	// Resize scroller's content area:
	NSSize		newSize = [taskContentView frame].size;
	newSize.height -= sizeGone.height;
	[taskContentView setFrameSize: newSize];
	[taskContentView setNeedsDisplay:YES];
	
	[unretainedTasks removeObject: theElement];
}


-(NSRect)	windowWillUseStandardFrame: (NSWindow*)sender defaultFrame: (NSRect)defaultFrame
{
	NSRect		theRect;
	
	// Leave at same position (if outside of screen it will placed on screen automatically)
	theRect.origin.x = [sender frame].origin.x;
	
	// y of origin has to adjusted with the height difference to leave the
	//  top of the window at the same place. 16 for the scrollbar and 17 for the
	//	status text
	theRect.origin.y = [sender frame].origin.y+[sender frame].size.height -([taskContentView frame].size.height +16+17);
	theRect.size.width = [sender frame].size.width;	
	
	// Calculate the optimal height (if larger than screen this will taken care of automatically)
	theRect.size.height = [taskContentView frame].size.height + 16+17;
	
	return theRect;
}


-(NSString*)	windowNibName
{
	#if UK_PROGRESS_PANEL_SAFARI_STYLE
	return @"UKListProgressPanel";
	#else
	return @"UKProgressPanel";
	#endif
}


@end


@implementation NSApplication (UKProgressPanel)

/* -----------------------------------------------------------------------------
	orderFrontProgressPanel:
		Category on NSApplication that adds a method for bringing the shared
		progress panel to front, creating it if there isn't one yet. You can
		use this as the action of a menu item (suggested name "Tasks") in your
		"Window" menu to allow that the user re-show the progress window once
		he has hidden it by clicking its close box.
   -------------------------------------------------------------------------- */

-(IBAction)			orderFrontProgressPanel: (id)sender
{
	[[UKProgressPanel sharedProgressPanel] orderFront: self];
}

@end
